# Advanced Charm Development

As you create more advanced charms, you'll find tips and tips here for making it a smoother process. There are a handful of tips that make development and repeat testing of charms a less time-ocn

# Juju

## Faster Deployments

When a charm is deployed, there are several time-consuming steps that are executed by default.

1. Launch an LXD container - download or update the cloud image for the series of charm being deployed
2. Run *apt-get update && apt-get upgrade*
3. Provision the machine with the Juju machine agent
4. Install charm (execute hooks, i.e., install, start)

### Build a custom cloud image

Caveat: This is intended only for use in a development environment, to provide faster iteration between deploying VNFs and charms.

The script below can be taken as-is We start with the base cloud image that LXD downloads from its [image server](https://us.images.linuxcontainers.org/), update it's installed software, and install the packages required by the reactive charm framework.

1. Launch a container using the latest cloud image
2. Run *apt-get update* and *apt-get upgrade*
3. Install extra packages needed by the reactive framework and your charm(s)
4. Publish the container as an image, under the alias *juju/$series/amd64*


**Note**: It's highly recommended to place this script into a nightly or weekly cron, so that you have relatively current updates.

```
#!/bin/bash
#
# This script will create trusty, xenial and/or bionic lxd images that will be used by the
# lxd provider in juju 2.1+ It is for use with the lxd provider for local
# development and preinstalls a common set of packages.
#
# This is important, as between them, basenode and layer-basic install ~111
# packages, before we even get to any packages installed by your charm.
#
# It also installs some helpful development tools, and pre-downloads some
# commonly used packages.
#
# This dramatically speeds up the install hooks for lxd deploys. On my slow
# laptop, average install hook time went from ~7min down to ~1 minute.
set -eux

# The basic charm layer also installs all the things. 47 packages.
LAYER_BASIC="gcc build-essential python3-pip python3-setuptools python3-yaml"

# the basic layer also installs virtualenv, but the name changed in xenial.
TRUSTY_PACKAGES="python-virtualenv"
XENIAL_PACKAGES="virtualenv"
BIONIC_PACKAGES="virtualenv"

# Predownload common packages used by your charms in development
DOWNLOAD_PACKAGES=

PACKAGES="$LAYER_BASIC $DOWNLOAD_PACKAGES"

function cache() {
    series=$1
    container=juju-${series}-base
    alias=juju/$series/amd64

    lxc delete $container -f || true
    lxc launch ubuntu:$series $container
    sleep 15  # wait for network

    lxc exec $container -- apt update -y
    lxc exec $container -- apt upgrade -y
    lxc exec $container -- apt install -y $PACKAGES $2
    lxc stop $container

    lxc image delete $alias || true
    lxc publish $container --alias $alias description="$series juju dev image ($(date +%Y%m%d))"

    lxc delete $container -f || true
}

# Uncomment the series you need pre-cached. By default, this will only
# cache the most recent series -- currently bionic.
# cache trusty "$TRUSTY_PACKAGES"
cache xenial "$XENIAL_PACKAGES"
# cache bionic "$BIONIC_PACKAGES"
```

### Disable OS upgrades

Prevent Juju from running *apt-get update && apt-get upgrade* when starting a machine

```
juju model-config enable-os-refresh-update=false enable-os-upgrade=false
```

### Using a custom Apt repository

You can configure Juju to use a local or regional Apt repository:

```
juju model-config apt-mirror=http://archive.ubuntu.com/ubuntu/
```

### Using a proxy server

Due to policy or network bandwidth, you may want to use a proxy server. Juju supports several types of proxy server, including:

- http-proxy
- https-proxy
- apt-http-proxy
- apt-https-proxy

```
juju model-config apt-http-proxy=http://squid.internal:3128 apt-https-proxy=https://squid.internal:3128
```

You can find a complete list of [model configuration](https://docs.jujucharms.com/2.4/en/models-config) keys in the [Juju Documentation](https://docs.jujucharms.com/2.4/en/).

## Debugging

[Debugging Charm Hooks](https://docs.jujucharms.com/2.4/en/developer-debugging) is a good place to start to familiarize yourself with the process and available ways of debugging a charm.

### Debug Logs

It's useful to watch the debug-logs while deploying a charm, to confirm what hooks are being run and to catch any exceptions that are raised. By default, it will tail the log for all charms:

```
$ juju debug-log
unit-charmnative-vnf-a-5: 18:12:11 INFO unit.charmnative-vnf-a/5.juju-log Reactive main running for hook start
unit-charmnative-vnf-a-5: 18:12:13 INFO unit.charmnative-vnf-a/5.juju-log Reactive main running for hook test
unit-charmnative-vnf-a-5: 18:12:13 INFO unit.charmnative-vnf-a/5.juju-log Invoking reactive handler: reactive/native-ci.py:21:test
unit-charmnative-vnf-a-5: 18:12:13 INFO unit.charmnative-vnf-a/5.juju-log Reactive main running for hook test
unit-charmnative-vnf-a-5: 18:12:13 INFO unit.charmnative-vnf-a/5.juju-log Invoking reactive handler: reactive/native-ci.py:21:test
unit-charmnative-vnf-a-5: 18:12:14 INFO unit.charmnative-vnf-a/5.juju-log Reactive main running for hook testint
unit-charmnative-vnf-a-5: 18:12:14 INFO unit.charmnative-vnf-a/5.juju-log Invoking reactive handler: reactive/native-ci.py:33:testint
unit-charmnative-vnf-a-5: 18:13:17 WARNING juju.worker.uniter.operation we should run a leader-deposed hook here, but we can't yet
unit-charmnative-vnf-a-5: 18:13:18 INFO unit.charmnative-vnf-a/5.juju-log Reactive main running for hook leader-settings-changed
unit-charmnative-vnf-a-5: 18:13:18 INFO unit.charmnative-vnf-a/5.juju-log Reactive main running for hook stop
```

### Interactive Debugging

One of the more useful, advanced tools we have is the *juju debug-hook* command, which lets us interact with the charm in a tmux session inside the container. This allows us to edit code and re-run it, use pdb, and inspect configuration and state. Please refer to the [Developer Debugging](https://docs.jujucharms.com/2.4/en/developer-debugging) docs for more information about how to do this.
